<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('video_info')) {
    // model pro info pro videa z Youtube, Vimeo, Vine. 
    class video_info {
        // ID videa
        public $id;
        // Domena videa (youtube, vimeo, vine)
        public $domain;
        // Thumbnaily v ruznych velikostech
        public $thumbnails = array();
        // V pripade chyby je typ chyby ulozen sem
        public $error;

    }

}

if (!class_exists('jwMedia')) {

    class jwMedia {

        /**
         * @name            parseJson
         * @description     vrati ID vsech obrazku obsazenych v JSONu
         * @param   string  $json
         * @return  array   
         */
        public static function parseJson($json) {
            $json = json_decode($json);
            $ids = array();
            foreach ((array) $json as $img) {
                if (isset($img->id)) {
                    $ids[] = $img->id;
                }
            }
            return $ids;
        }

        /**
         * @name            getPostThumbnail
         * @description     Vyrenderuje IMG tag pro post featured imagre s případným odkazem (dokaze odkazat na prettyphoto) nebo lazyload. Nshrada za the_post_thumbnail()
         * 
         * @param type $post_id
         * @param type $sizes
         * @param type $link
         * @param string $prettyPhoto
         * @param type $attr
         * @return string
         */
        public static function getPostThumbnail($post_id, $sizes = array(), $link = false, $prettyPhoto = array(), $attr = array()) {
            $out = '';

            if (has_post_thumbnail($post_id)) {
                $thumbnailId = get_post_thumbnail_id($post_id);
                $out .= self::getAttachment($thumbnailId, $sizes, $link, $prettyPhoto, $attr);
            }
            return $out;
        }

        /**
         * @name            getPostThumbnailSrc
         * @description     Vraci SRC featured image attachmentu 
         * 
         * @param type $post_id
         * @param type $size
         * @return type
         */
        public static function getPostThumbnailSrc($post_id, $size = 'full') {
            $src = '';
            if (has_post_thumbnail($post_id)) {
                $thumbnailId = get_post_thumbnail_id($post_id);
                $src = self::getAttachmentSrc($thumbnailId, $size);
            }
            return $src;
        }

        /**
         * @name            getAttachmentSrc
         * @description     Vraci SRC daného attachmentu
         * 
         * @param type $attachment_id
         * @param type $size
         * @return type
         */
        public static function getAttachmentSrc($attachment_id, $size = 'full') {
            $src = array();
            if (isset($attachment_id) && $attachment_id > 0) {
                $src = wp_get_attachment_image_src($attachment_id, $size);
            }
            if (isset($src[0])) {
                return $src[0];
            }
            return '';
        }

        /**
         * @name            getAttachment
         * @description     Vyrenderuje IMG tag pro attachment imagre s případným odkazem (dokaze odkazat na prettyphoto) nebo lazyload.
         * 
         * @param int           $attachment_id
         * @param array         $sizes
         * @param bool/string   $link
         *                      string   $link           -> string odkazu
         *                      bool     $link = true    -> odkaz vede na obrazek
         * @param array         $prettyPhoto 
         *                      [0] bool    Pouzit prettyphoto
         *                      [1] string  Velikost prettyphoto obrazku
         *                      [2] string  Skupina obrazku (galerie prettyPhoto)   
         * @param array         $attr
         * @return string
         */
        public static function getAttachment($attachment_id, $sizes = array(), $link = false, $prettyPhoto = array(), $attr = array()) {
            $out = '';
            $originalattr = $attr;

            if (empty($attachment_id)) {
                return '';
            }

            if ($link) {  //pokud chci obrázek pouzit jako odkaz
                $out = '<a title="' . get_the_title(get_the_ID()) . '" ';
                if (isset($prettyPhoto[0]) && $prettyPhoto[0]) {  //pro pouziti prettyPhota
                    if (isset($prettyPhoto[1])) {
                        $ppSize = $prettyPhoto[1];
                    } else {
                        $ppSize = 'full';
                    }
                    if (isset($prettyPhoto[2])) {
                        $ppId = $prettyPhoto[2];
                    } else {
                        $ppId = 'pretty';
                    }
                    if($link === true){
                        $link = wp_get_attachment_image_url($attachment_id, $ppSize);
                    }
                    $out .= 'href="' . esc_url($link) . '" ';
                    $out .= 'rel="prettyPhoto[' . esc_attr($ppId) . ']"';
                } else { // klasický odkaz
                    if($link === true){
                        $link = wp_get_attachment_image_url($attachment_id, 'full');
                    }
                    $out .= 'href="' . esc_url($link) . '" ';
                }
                $out .= '>';
            }
            
            $respImg = self::getImageSources($attachment_id, $sizes);

            if(!isset($attr['class'])){
                $attr['class'] = '';
            }
            $attr['class'] = $attr['class'] . ' ' . 'jw-responsive-img';   
            $attr = array_merge($attr, array('data-src' => json_encode($respImg)));
            if (jwOpt::get_option('scroll_load_images', '0')) {
                $attr['class'] = $attr['class'] . ' ' . 'jaw-hide-image';   
                $attr = array_merge($attr, array("src" => "data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7")); //LAZYLOAD - zakladni obrazek bude pruhledny o velikosti 1px x 1px. Požadovaná velikost se načte až pomocí javascriptu.
            }
            
            $out .= wp_get_attachment_image($attachment_id, $sizes[0], false, $attr);
            
            if (jwOpt::get_option('scroll_load_images', '0') != '0') {
                $out .= '<noscript>';
                $originalattr = array_merge($originalattr, array('itemprop' => 'contentUrl'));
                $out .= wp_get_attachment_image($attachment_id, $sizes[0], false, $originalattr);
                $out .= '</noscript>';
            }
            if ($link) {
                $out .= '</a>';
            }
            
            return $out;
        }

        /**
        * @name             getImageSources
        * @description      vrati src a width pro vsechny obrazyk, ktere chci pouzit pro responzivni loading
        *
        */
        public static function getImageSources($attachment_id, $sizes = array()) {
            $allSizes = wp_get_attachment_metadata($attachment_id);
            if(isset($allSizes['sizes']) && !isset($allSizes['sizes']['full'])){
                $allSizes['sizes']['full'] = array($allSizes['file'], $allSizes['width'], $allSizes['height'], false);    
            }
            if (empty($sizes)) {
                $sizes[] = 'full';
            }
            $respImg = array();
            foreach ((array) $sizes as $size) {
                if (isset($allSizes['sizes']) && in_array($size, array_keys($allSizes['sizes']))) {
                    $respImg[$size] = $allSizes['sizes'][$size];
                    $file = wp_get_attachment_image_src($attachment_id, $size);
                    $respImg[$size]['file'] = $file[0];
                    if (isset($respImg[$size]['height']))
                        unset($respImg[$size]['height']);
                    if (isset($respImg[$size]['mime-type']))
                        unset($respImg[$size]['mime-type']);
                }
            }
            $respImg = jwUtils::sortByWidth($respImg);
            return $respImg;
        }

        /**
        * @name             getVideoThumbnail
        * @description      vrati tumbnaily k danemu videu
        *
        * @param    string  $url
        * @param    boolean $link
        * @param    array   $prettyPhoto
        * @param    array   $attr  
        */
        public static function getVideoThumbnail($url, $link = false, $prettyPhoto = array(), $attr = array()) {
            $out = '';
            $video_infos = self::getVideoInfo($url);
            if(isset($video_infos->error)){
                $out .= jwUtils::jaw_msg(sprintf(esc_html__('Video Player Error: %s','gdn-theme'),$video_infos->error));
                return $out;
            }
            if (isset($video_infos->id)) {

                if ($link) {  //pokud chci obrázek pouzit jako odkaz
                    $out = '<a title="' . get_the_title(get_the_ID()) . '" ';
                    if (isset($prettyPhoto[0]) && $prettyPhoto[0]) {  //pro pouziti prettyPhota
                        if (isset($prettyPhoto[1])) {
                            $ppSize = $prettyPhoto[1];
                        } else {
                            $ppSize = 'full';
                        }
                        if (isset($prettyPhoto[2])) {
                            $ppId = $prettyPhoto[2];
                        } else {
                            $ppId = 'pretty';
                        }

                        $out .= 'href="' . esc_url($url) . '" ';
                        $out .= 'rel="prettyPhoto[' . esc_attr($ppId) . ']"';
                    } else { // klasický odkaz
                        $out .= 'href="' . esc_url($link) . '" ';
                    }
                    $out .= '>';
                }
                $respImg = array();
                $allSizes = $video_infos->thumbnails;
                $sizes = array('100', '200', '480');
                if (sizeof($allSizes) > 0) {
                    $i = 0;
                    foreach ((array) $allSizes as $key => $size) {
                        $respImg[$key] = array(
                            'file' => $size,
                            'width' => $sizes[$i++]
                        );
                    }
                    $attr = array_merge($attr, array('data-src' => json_encode($respImg), 'class' => 'jw-responsive-img'));
                }
                $attr = array_map('esc_attr', $attr);
                if(isset($allSizes['thumbnail_medium'])){
                    $out .= '<img src="' . esc_url($allSizes['thumbnail_medium']) . '" alt="Video" ';
                    foreach ($attr as $key => $value) {
                        $out .= esc_attr($key) . '="' . esc_attr($value) . '" ';
                    }
                    $out .= ' />';
                }
                if ($link) {
                    $out .= '</a>';
                }
            }
            return $out;
        }
        
        
        public static function getResponsiveImage($mainImg, $reposiveImgs = array(), $attr = array()){
            
            $out = "";
            if(!isset($attr['class'])){
                $attr['class'] = "";
            }
            
            $attr['class'] .= ' jw-responsive-img ';   
            $reposiveImgs = jwUtils::sortByWidth($reposiveImgs);
            $attr['data-src'] = json_encode($reposiveImgs);
            $attr = array_map('esc_attr', $attr);
            if (jwOpt::get_option('scroll_load_images', '0')) {
                $attr['class'] .= ' jaw-hide-image ';   
                $attr["src"] = "data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7";
            }else{
                $attr["src"] = $mainImg;
            }
            
            $out .= "<img ";
            foreach ($attr as $key => $value) {
                $out .= esc_attr($key) . '="' . esc_attr($value) . '" ';
            }
            $out .= "/>";
            
            return $out;
        }

        /**
         * @name            getVideoPlayer
         * @description     Na zaklade url vygeneruje html (iframe) prehravace.
         */
        public static function getVideoPlayer($url, $width) {
            $out = '';
            $video = self::getVideoInfo($url);
            $height = (int) ($width / 1.78); //über konstanta
            if(isset($video->error)){
                $out .= jwUtils::jaw_msg(sprintf(esc_html__('Video Player Error: %s','gdn-theme'),$video->error));
                return $out;
            }
            if ($video->domain == 'youtube') {
                $out .= '<iframe  class="video jaw-keep-ratio" itemscope itemtype="//schema.org/VideoObject" width="100%" height="' . esc_attr($height) . '" src="'.esc_url('//www.youtube.com/embed/' . $video->id) . '" frameborder="0" allowfullscreen>';
                if(isset($video->thumbnails['thumbnail_medium'])){
                    $out .= '<meta itemprop="thumbnailUrl" content="' . esc_url($video->thumbnails['thumbnail_medium']) . '" />';
                }
                $out .= '<meta itemprop="embedURL" content="'.esc_url('//www.youtube.com/embed/' . $video->id) . '" />                
                    </iframe>';
            } else if ($video->domain == 'vimeo') {
                $out .= '<iframe class="video jaw-keep-ratio" itemscope itemtype="//schema.org/VideoObject" src="'.esc_url('//player.vimeo.com/video/' . $video->id) . '" width="100%" height="' . esc_attr($height) . '" frameborder="0" webkitAllowFullScreen mozallowfullscreen allowFullScreen>';
                if(isset($video->thumbnails['thumbnail_medium'])){
                    $out .= '<meta itemprop="thumbnailUrl" content="' . esc_url($video->thumbnails['thumbnail_medium']) . '" />';
                }
                $out .= '<meta itemprop="embedURL" content="'.esc_url('//player.vimeo.com/video/' . $video->id) . '" />                
                    </iframe>';
            } else if ($video->domain == 'vine') {
                $out .= '<iframe class="video vine-embed jaw-keep-ratio" src="'.esc_url('//vine.co/v/' . $video->id) . '/embed/simple" width="100%" height="' . esc_attr($width) . '" frameborder="0" ></iframe>';
            }
            else if ($video->domain == 'dailimotion') {
                $out .= '<iframe class="video dailimotion-embed jaw-keep-ratio" src="'.esc_url('//www.dailymotion.com/embed/video/' . $video->id) . '" width="100%" height="' . esc_attr($height) . '" frameborder="0" ></iframe>';
            }
            else if ($video->domain == 'facebook') {
                $out .= '<div class="fb-video jaw-keep-ratio"
                        data-href="'.$url.'"
                        data-width="1500"
                        data-allowfullscreen="true"
                        data-show-captions="true"></div>';
            }
            else { //self hosted video
                $out .= '<video class="video selfhosted jaw-keep-ratio" controls width="100%" height="' . esc_attr($height) . '"  >';
                $out .= '<source src="'. $video->id .'" type="video/mp4">';
                $out .= 'Your browser does not support the video tag.';
                $out .= '</video>';
            }
            return $out;
        }

        /** 
         * @name            get_video_info
         * @description     vrací z url adresy (např: http://vimeo.com/26609463) jen ID, název domény a náhledy ve 3 velikostech.
         */
        public static function getVideoInfo($video_url, $thumbnails = true) {
            $ret = new video_info();
            if (preg_match('%(?:youtube(?:-nocookie)?\.com/(?:[^/]+/.+/|(?:v|e(?:mbed)?)/|.*[?&]v=)|youtu\.be/)([^"&?/ ]{11})%i', $video_url, $match)) {
                $ret->id = $match[1];
                $ret->domain = 'youtube';
                if ($thumbnails) {
                    $ret->thumbnails['thumbnail_small'] = "https://img.youtube.com/vi/" . $ret->id . "/default.jpg";
                    $ret->thumbnails['thumbnail_medium'] = "https://img.youtube.com/vi/" . $ret->id . "/mqdefault.jpg";
                    $ret->thumbnails['thumbnail_large'] = "https://img.youtube.com/vi/" . $ret->id . "/hqdefault.jpg";
                }
            } else if (preg_match('/.*?vimeo\.com\/(clip\:|.*\/)?(\d+).*$/', $video_url, $match)) {
                $ret->id = $match[2];
                $ret->domain = 'vimeo';
                //u vimea jde do thumbnails více informací (např: description, date, nuber of likes ...)
                $thumbnails = wp_remote_request("https://vimeo.com/api/v2/video/" . $ret->id . ".php");
                //osetreni v pripade nedostupnosti serveru
                if($thumbnails instanceof WP_Error){
                    $ret->error = esc_html__('Vimeo server problem','gdn-theme');
                    return $ret;
                }
                if(isset($thumbnails['body'])){
                    $thumbnails = maybe_unserialize($thumbnails['body']);
                }
                if (isset($thumbnails[0]['thumbnail_small']) && isset($thumbnails[0]['thumbnail_medium']) && isset($thumbnails[0]['thumbnail_large'])) {
                    $ret->thumbnails['thumbnail_small'] = $thumbnails[0]['thumbnail_small'];
                    $ret->thumbnails['thumbnail_medium'] = $thumbnails[0]['thumbnail_medium'];
                    $ret->thumbnails['thumbnail_large'] = $thumbnails[0]['thumbnail_large'];
                }else{
                    $ret->error = $thumbnails;
                }
            } else if (preg_match('/^http(s)?:\/\/(www\.)?vine\.co\/v\/(.*?)(\/.*)?$/', $video_url, $match)) {
                $ret->id = $match[3];
                $ret->domain = 'vine';
                if ($thumbnails) {
                    $vine = wp_remote_request("https://vine.co/v/{$ret->id}");
                    //osetreni v pripade nedostupnosti serveru
                     if($vine instanceof WP_Error){
                        $ret->error = esc_html__('Vine server problem','gdn-theme');
                        return $ret;
                    }
                    $vine = wp_remote_retrieve_body($vine);
                    preg_match('/property="og:image" content="(.*?)"/', $vine, $matches);
                    $ret->thumbnails['thumbnail_medium'] = ($matches[1]) ? $matches[1] : false;
                }
            }
            else if(preg_match('/^http(s)?:\/\/(www\.)?dai\.ly\/(.*?)(\/.*)?$/', $video_url, $match)) {
                $ret->id = $match[3];
                $ret->domain = 'dailimotion';
                if ($thumbnails) {
                    $ret->thumbnails['thumbnail'] = "http://www.dailymotion.com/thumbnail/video/" . $ret->id;
                }
            }
            else if(preg_match('/^http(s)?:\/\/(www\.)?facebook\.com\/(.*?)\/videos\/(.*)?(\/)?$/', $video_url, $match)) {
                $ret->id = $match[4];
                $ret->domain = 'facebook';
                if ($thumbnails) {
                    $ret->thumbnails['thumbnail'] = "https://graph.facebook.com/" . $ret->id . "/picture";
                }
            }
             else {
                $ret->id = $video_url;
            }
            return $ret;
        }
        

        /**
        * @name             getGalleryID
        * @description      Returns ID of a gallery
        * @return           Int
        */
        private static function getGalleryID() {
            global $post;
            $post_meta = get_post_meta($post->ID, '');
            $gallery_box_id = "";
            if (isset($post_meta)) {
                $gallery_box_id = jaw_template_get_var('gallery');
            }
            if(empty($gallery_box_id)) {
                if (isset($post_meta['jaw_perex_gallery_box'][0])) {
                    $gallery_box_id = (int) $post_meta['jaw_perex_gallery_box'][0];
                }
            }
            
            return $gallery_box_id;
        }
        
        /**
        * @name             getGallery
        * @description      Returns number 1 / 0 if gallery is valid
        * @return           Int
        */
        private static function getGalleryValidation() {
            $gallery_id = self::getGalleryID();
            $gallery_args = array(
            'post_type' => 'jaw-gallery',
            'numberposts' => '1',
            'post__in' => array($gallery_id)
            );
            $is_gallery_valid = get_posts($gallery_args);
            $force_gallery_id = 0;
            if (!empty($is_gallery_valid) || $gallery_id == 0 || $gallery_id == '') {
                $force_gallery_id = 1;
            }
            
            return $force_gallery_id;
        }
        
        /**
        * @name             getGalleryContent
        * @description      Returns number 1 / 0 if gallery is valid
        * @return           Array
        */
        private static function getGalleryContent($id="",$thumbs="",$type="") {
            if(empty($id)) {
                $gallery_box_id = self::getGalleryID();
            } else {
                $gallery_box_id = $id; // widget
            }
            $single_preset = jaw_template_get_var('single_preset');
            if(!empty($thumbs)) {
                $gallery_count = $thumbs;
            } else {
                $gallery_count = jwOpt::getPresetValue('perex_gallery', '4', 'single', $single_preset);
            }
            if($type == "horizontal") {
                switch($gallery_count) {
                    case "less" :
                    case 4 :
                        $gallery_count = 5;
                    break;
                    case "more" :
                    case 6 :
                        $gallery_count = 7;
                    break;
                }
            } else {
                switch($gallery_count) {
                    case "less" :
                    case 4 :
                        $gallery_count = 4;
                    break;
                    case "more" :
                    case 6 :
                        $gallery_count = 6;
                    break;
                }
            }
            $gallery_url = add_query_arg('poid', get_the_ID(), get_permalink($gallery_box_id));
            $carousel = array();
            $gallery_imgs_url = array();
            $gallery_types = array();
            $video_embed_set = 0;
            $limit=0;
            $gal_max_count=0;
            $video_max_count=0;
            $embed_max_count=0;
            // items count for gallery bottom text
            $items_name = "";
            $items_count = "";
            if (self::getGalleryValidation()) { // is valid
                // images
                $jaw_gallery_items = get_post_meta($gallery_box_id, 'jaw_gallery_items', true);
                $gal = jwMedia::parseJson($jaw_gallery_items);
                $gal_max_count = count($gal);
                if (isset($gal)) {
                    $carousel['gallery'] = array();
                    foreach ((array) $gal as $i) {
                        if($limit < $gallery_count) {
                            $carousel['gallery'][] .= jwMedia::getAttachment($i, array('thumbnail', 'jaw_blog-special', 'jaw_slider-square'));
                            $limit++;
                        }
                    }
                }
                // embed stuff
                $_embed_items = get_post_meta($gallery_box_id, 'jaw_gallery_embed_link', true);
                $_embed_items_parsed = json_decode($_embed_items, true);
                $embed_max_count = count($_embed_items_parsed);
                $_gallery_embed_pic = jwOpt::get_option('gallery_embed_pic', 'def');
                // Custom embed icon for embed stuff in gallery
                if (!empty($_gallery_embed_pic)) {
                    $parse_custom_img = (json_decode($_gallery_embed_pic, true));
                    $_gallery_embed_pic_url = jwMedia::getAttachment($parse_custom_img[0]['id'], array('thumbnail'));
                } else {
                    $_gallery_embed_pic_url = '<i class="jaw-icon-embed3"></i>'; // default img
                }
                $carousel['embed_link'] = array();
                if (isset($_embed_items_parsed) && !empty($_embed_items_parsed)) {
                    foreach ((array) $_embed_items_parsed as $k => $i) {
                        if(!empty($_embed_items_parsed[$k])) {
                            if (!is_null($i['jaw_gallery_embed_link']) && !empty($i['jaw_gallery_embed_link'])) {
                                if($limit < $gallery_count) {
                                    $carousel['embed_link'][$k] = '<div class="item-wrapper embed_link"><div class="embed_container">';
                                    $carousel['embed_link'][$k] .= $_gallery_embed_pic_url;
                                    $carousel['embed_link'][$k] .= '</div></div>';
                                    $limit++;
                                }
                            }
                        }
                    }
                    $video_embed_set = 1;
                }
                // video url
                $_url_items = get_post_meta($gallery_box_id, 'jaw_gallery_video_link', true);
                $_url_items_parsed = json_decode($_url_items, true);
                $video_max_count=count($_url_items_parsed);
                $carousel['video_link'] = array();
                if (isset($_url_items_parsed) && !empty($_url_items_parsed)) {
                    foreach ((array) $_url_items_parsed as $k => $i) {
                        if(!empty($_url_items_parsed[$k])) {
                            if (!is_null($i['jaw_gallery_video_link']) && !empty($i['jaw_gallery_video_link'])) {
                                if($limit < $gallery_count) {
                                $video_thumb = jwMedia::getVideoInfo($_url_items_parsed[$k]['jaw_gallery_video_link']);
                                if(isset($video_thumb->thumbnails['thumbnail_medium'])){
                                        $carousel['video_link'][$k] = '<span class="jaw-video-icon"><i class="jaw-icon-play4"></i></span><div class="item-wrapper">';
                                        $carousel['video_link'][$k] .= '<img src="' . esc_url($video_thumb->thumbnails['thumbnail_medium']) . '"  alt="Video" / >';
                                        $carousel['video_link'][$k] .= '</div>';
                                        $limit++;
                                    }
                                }
                            }
                        }
                    }
                    $video_embed_set = 1;
                }
                // types position
                $gallery_types = array(
                    0 => "gallery",
                    1 => "video_link",
                    2 => "embed_link"
                );
                switch ($video_embed_set) {
                    case 1 :
                        $items_count = $gal_max_count+$video_max_count+$embed_max_count;
                        $items_name = " Items";
                        break;
                    default:
                        $items_count = $gal_max_count;
                        $items_name = " Photos";
                        break;
                }
                $galery_imgs_count = 0;
                foreach ($gallery_types as $gtypes) {
                    $galery_imgs_count += isset($carousel['' . $gtypes . '']) ? count($carousel['' . $gtypes . '']) : 0;
                    if (empty($carousel['' . $gtypes . ''])) {
                        $carousel['' . $gtypes . ''] = array();
                    }
                }
                for ($i = 0; $i < $galery_imgs_count; $i++) {
                    $gallery_imgs_url[$i] = add_query_arg('imid', $i + 1, $gallery_url);
                }
            }
            // THUMBS CHANGE HERE
            $galleryVars[] = array(
                "galleryImgUrl" => $gallery_imgs_url,
                "galleryCount" => $gallery_count,
                "galleryItemsCount" => $items_count,
                "galleryItemsName" => $items_name,
                "galleryPreset" => $single_preset,
                "galleryTypes" => $gallery_types,
                "galleryUrl" => $gallery_url
            );
            $returnGallery = array_merge($galleryVars,$carousel);
            
            return $returnGallery;
        }
        
        private static function getGalleryView($type,$gallery_count,$carousel,$gallery_types,$gallery_imgs_url,$gallery_url,$single_preset,$items_count,$items_name,$id="") {
            if(empty($id)) {
                $gallery_box_id = self::getGalleryID();
            } else {
                $gallery_box_id = $id; // widget
            }
            $out = "";
            $content = "";
            $count = 0;
            $col_max = "";
            $gallery_class = "";
            jaw_template_inc_counter('gallery');
            $gallery_default_name = apply_filters("jaw_gallery_box_default_name","gallery");
            $gallery_box_default_items_name = apply_filters("jaw_gallery_box_default_items_name",$items_name);
            $gallery = array_merge(array("gallery"=>$carousel["gallery"],"video_link"=>$carousel["video_link"],"embed_link"=>$carousel["embed_link"]));
            switch($type) {
                case "vertical" :
                    // vertical
                    switch ($gallery_count) {
                        case 6 :
                        case "more" :
                            $col_max = 4;
                            $gallery_class = "col-mob-6";
                            $gallery_count = 6;
                            break;
                        case 4 :
                        case "less" :
                            $col_max = 3;
                            $gallery_class = "col-mob-4";
                            $gallery_count = 4;
                            break;
                        default:
                            $col_max = 12;
                            $gallery_class = "";
                    }
                    if (!empty($gallery)) {
                        for ($a = 0; $a < count($gallery); $a++) {
                            foreach ((array) $gallery['' . $gallery_types[$a] . ''] as $car) {
                                foreach ((array) $car as $c) {
                                    if ($count < $gallery_count) {
                                        $content .= '<div style="padding: 0" class="col-lg-6 col-md-6 col-sm-' . esc_attr($col_max ). ' col-xs-' . esc_attr($col_max) . ' item-box ' . esc_attr($gallery_class) . '">';
                                        $content .= '<a target="_self" href="' . esc_url($gallery_imgs_url[$count]) . '">';
                                        $content .= $c;
                                        $content .= '</a>';
                                        $content .= '</div>';
                                        $count++;
                                    }
                                }
                            }
                        }
                    }
                    // view
                    $out = "";
                    $out .= '<div class="jaw-media-gallery-box gallery-vertical">';
                        $out .= '<div class="row">';
                            $out .= '<div class="col-xs-12 col-sm-12 col-md-12 col-lg-12">';
                                $out .= '<div class="jaw-gallery-title"><a href="'.esc_url($gallery_url).'">'.$gallery_default_name.'<i class="jaw-icon-arrow-right3"></i></a></div>';
                            $out .= '</div> ';
                        $out .= '</div> ';
                        $out .= '<div class="row">';
                            $out .= '<div class="col-xs-12 col-sm-12 col-md-12 col-lg-12">';
                                $out .= '<div id="jaw-gallery-box'.esc_attr( jaw_template_get_counter('gallery')).'" class="jaw-gallery">';
                                    $out .= '<div class="carousel-inner">';
                                        $out .= $content;
                                    $out .= '</div>';
                                    $out .= '<div class="jaw-gallery-bottom-text"><a target="_self" href="'.esc_url($gallery_url).'">';
                                        $out .= jwUtils::crop_length(get_the_title($gallery_box_id), jwOpt::getPresetValue('perex_gallery_title_text', '-1', 'single', $single_preset)).'</a>';
                                    $out .= '</div>';
                                $out .= '</div>';
                            $out .= '</div>';
                        $out .= '</div>';
                    $out .= '</div>';
                    
                    return $out;
                break;
                case "horizontal" :
                    // horizontal
                    switch ($gallery_count) {
                        case 6 :
                        case "more" :
                            $col_max = 2;
                            $gallery_class = "col-lg-12-7 col-mob-6";
                            $gallery_count = 7;
                            break;
                        case 4 :
                        case "less" :
                            $col_max = 3;
                            $gallery_class = "col-lg-12-5 col-mob-4";
                            $gallery_count = 5;
                            break;
                        default:
                            $col_max = 12;
                            $gallery_class = "";
                    }
                    if (!empty($gallery)) {
                        for ($a = 0; $a < count($gallery); $a++) {
                            foreach ((array) $gallery['' . $gallery_types[$a] . ''] as $car) {
                                foreach ((array) $car as $c) {
                                    if ($count < $gallery_count) {
                                        $content .= '<div class="col-lg-' . esc_attr($col_max) . ' col-md-2 col-sm-4 col-xs-4 item-bottom ' . esc_attr($gallery_class) . '">';
                                        $content .= '<a target="_self" href="' . esc_url($gallery_imgs_url[$count]) . '">';
                                        $content .= $c;
                                        $content .= '</a>';
                                        $content .= '</div>';
                                        $count++;
                                    }
                                }
                            }
                        }
                    }
                    // view
                    $out = "";
                    $out .= '<div class="row jaw-media-gallery-bottom">';
                        $out .= '<div class="col-xs-12 col-sm-12 col-md-12 col-lg-12">';
                            $out .= '<div class="jaw-gallery-title"><a target="_self" href="'.esc_url($gallery_url).'">'.$gallery_default_name.'<i class="jaw-icon-arrow-right3"></i></span></div></a>';
                        $out .= '</div>';
                        $out .= '<div class="col-xs-12 col-sm-12 col-md-12 col-lg-12">';
                            $out .= '<div id="jaw-gallery-bottom-'.esc_attr(jaw_template_get_counter('gallery')).'" class="jaw-gallery jaw-gal-container">';
                                $out .= '<div class="carousel-inner">';
                                    $out .= '<div class="jaw-gallery-bottom-text">';
                                    $out .= '</div></a>';
                                    $out .= $content;
                                    $out .= '</div>';
                                $out .= '</div>';
                            $out .= '</div>';
                            $out .= '<div class="col-xs-12 col-sm-12 col-md-12 col-lg-12">';
                                $out .= '<div class="jaw-gallery-title-bottom">';
                                    $out .= '<a target="_self" href="'.esc_url($gallery_url).'"><span>'.jwUtils::crop_length(get_the_title($gallery_box_id), jwOpt::getPresetValue('perex_gallery_bottom_title_text', '-1', 'single', $single_preset)).'</span> - <span>'.$items_count.' '.$gallery_box_default_items_name.'</span></a>';
                                $out .= '</div>';
                            $out .= '</div>';
                        $out .= '</div>';
                        
                    return $out;
                break;
                case "single" :
                    $img_content = json_decode(get_post_meta($gallery_box_id, 'jaw_gallery_image', true));
                    $out = "";
                    if($img_content) {
                        $imgId = $img_content[0]->id;
                        if(!empty($imgId)) {
                            $out .= '<div class="media-gallery-item">';
                                $out .= '<a href="'.esc_url($gallery_imgs_url[0]).'">';
                                    $out .= jwMedia::getAttachment($imgId, array( 'middle', 'large'));//responsive image
                                        $out .= '<div class="jaw-gallery-title">'.$gallery_default_name.'<span class="jaw-photo-count">+'.$items_count.''. $gallery_box_default_items_name.'</span>';
                                    $out .= '</div>';
                                $out .= '</a>';
                            $out .= '</div>';
                        } else {
                            return jwUtils::jaw_msg(esc_html__("You have not set a gallery thumbnail!",'gdn-theme'));
                        }
                    } else {
                        return jwUtils::jaw_msg(
                            sprintf(esc_html__('You have probably not set a gallery thumbnail or gallery ID does not exists ( %s ). Please %sEDIT%s this gallery.','gdn-theme'),$gallery_box_id,'<a href="'.get_edit_post_link($gallery_box_id).'"#section-_perex_gallery_box>','</a>'));
                    }
                    return $out;
                break;
                case "special" :
                    if (!empty($gallery) && !empty($gallery_imgs_url)) {
                        $max_img_print = 4;
                        for ($a = 0; $a < count($gallery); $a++) {
                            foreach ((array) $gallery['' . $gallery_types[$a] . ''] as $car) {
                                foreach ((array) $car as $c) {
                                    if ($count < $max_img_print) {
                                        $content .=  '<div class="item-bottom">';
                                        $content .=  '<a target="_self" href="' . esc_url($gallery_imgs_url[$count]) . '">';
                                        $content .=  $c;
                                        $content .=  '</a>';
                                        $content .=  '</div>';
                                        $count++;
                                    }
                                }
                            }
                        }
                        $out = "";
                        $out .= '<div class="content-special-images-container">';
                            $out .= '<div class="content-special-images">';
                                $out .= '<div id="jaw-gallery-bottom-'.esc_attr( jaw_template_get_counter('gallery')).'" class="jaw-gallery jaw-gal-container">';
                                    $out .= '<div class="img-list">'.$content.'</div>';
                                $out .= '</div>';
                            $out .= '</div>';
                        $out .= '</div>';
                    }
                    return $out;
                break;
            }
        }
        
        /**
        * @name             getGallery
        * @description      Returns final gallery view
        * @return           String
        */
        public static function getGallery($type="vertical",$thumbs="",$id="") {
            $gallery_vars = self::getGalleryContent($id,$thumbs,$type);
            if(empty($id)) {
                $gallery_box_id = self::getGalleryID();
            } else {
                $gallery_box_id = $id; // widget
            }
            if(!empty($thumbs)) {
                $gallery_bottom_count = $thumbs;
            } else {
                $gallery_bottom_count = jwOpt::getPresetValue('post_gallery_bottom', '4', 'single', $gallery_vars[0]["galleryPreset"]);
            }
            if (!self::getGalleryValidation()) {
                return jwUtils::jaw_msg(sprintf(esc_html__("The gallery ID does not exist: %s Please %sedit%s this post and set proper gallery ID",'gdn-theme'),$gallery_box_id,'<a href='.get_edit_post_link($gallery_box_id).'#section-_perex_gallery_box>','</a>'));
            }
            if ($gallery_box_id != null && $gallery_vars[0]["galleryItemsCount"] != '0' && self::getGalleryValidation()) {
                return self::getGalleryView(
                    $type,
                    $gallery_bottom_count,
                    $gallery_vars,
                    $gallery_vars[0]["galleryTypes"],
                    $gallery_vars[0]["galleryImgUrl"],
                    $gallery_vars[0]["galleryUrl"],
                    $gallery_vars[0]["galleryPreset"],
                    $gallery_vars[0]["galleryItemsCount"],
                    $gallery_vars[0]["galleryItemsName"],
                    $gallery_box_id
                    );
            }
            return;
        }
    }
}